<?php
namespace App\Services;

use App\Config\Database;
use PDO;

class RAGService {
    private string $apiKey;
    
    public function __construct() {
        // En producción usar variables de entorno: getenv('GEMINI_API_KEY')
        $this->apiKey = 'TU_CLAVE_API_GEMINI_AQUI'; 
    }

    /**
     * 1. Buscar documentos en base Legacy (Latin1)
     */
    public function searchContext(string $query): array {
        $db = Database::getLegacy();
        
        $sql = "SELECT id_documento, titulo, contenido, archivotxt, resumen, fecha1, emisor,
                MATCH(titulo, contenido) AGAINST (:query IN NATURAL LANGUAGE MODE) as score
                FROM documento
                WHERE estado = 1 AND vigencia = 1 AND desplegado = 1
                AND MATCH(titulo, contenido) AGAINST (:query IN NATURAL LANGUAGE MODE)
                ORDER BY score DESC LIMIT 5";

        $stmt = $db->prepare($sql);
        $stmt->bindValue(':query', $query, PDO::PARAM_STR);
        $stmt->execute();
        
        $rawResults = $stmt->fetchAll();
        $processed = [];

        foreach ($rawResults as $row) {
            // Lógica de fallback de contenido
            $textoBase = !empty($row['contenido']) ? $row['contenido'] : $row['archivotxt'];
            
            // CONVERSIÓN CRÍTICA: Latin1 -> UTF-8
            $titulo = mb_convert_encoding($row['titulo'], 'UTF-8', 'ISO-8859-1');
            $contenido = mb_convert_encoding($textoBase, 'UTF-8', 'ISO-8859-1');
            $resumen = mb_convert_encoding($row['resumen'] ?? '', 'UTF-8', 'ISO-8859-1');
            $emisor = mb_convert_encoding($row['emisor'] ?? '', 'UTF-8', 'ISO-8859-1');

            // Limpiar HTML y recortar para evitar exceso de tokens
            $contenidoLimpio = strip_tags($contenido);
            $contenidoLimpio = substr($contenidoLimpio, 0, 8000); // ~2000 tokens aprox por doc

            $processed[] = [
                'id' => $row['id_documento'],
                'titulo' => $titulo,
                'fecha' => $row['fecha1'],
                'emisor' => $emisor,
                'resumen' => $resumen,
                'contenido' => $contenidoLimpio,
                'score' => $row['score']
            ];
        }

        return $processed;
    }

    /**
     * 2. Consultar Gemini API
     */
    public function queryGemini(string $userQuery, array $contextDocs): array {
        $endpoint = "https://generativelanguage.googleapis.com/v1beta/models/gemini-1.5-flash:generateContent?key=" . $this->apiKey;

        // Construir Contexto
        $contextStr = "";
        foreach ($contextDocs as $doc) {
            $contextStr .= "DOC ID: {$doc['id']} | TÍTULO: {$doc['titulo']} | FECHA: {$doc['fecha']} | EMISOR: {$doc['emisor']}\n";
            $contextStr .= "RESUMEN: {$doc['resumen']}\n";
            $contextStr .= "CONTENIDO: {$doc['contenido']}\n\n----------------\n\n";
        }

        $systemPrompt = "Eres un abogado experto en legislación ecuatoriana. " .
            "Responde basándote EXCLUSIVAMENTE en los fragmentos de documentos proporcionados. " .
            "Cita siempre el Título y Fecha del documento fuente. " .
            "Si la respuesta no está en los documentos, indícalo claramente.";

        $payload = [
            "contents" => [
                ["parts" => [["text" => "CONTEXTO LEGAL:\n" . $contextStr . "\n\nPREGUNTA USUARIO:\n" . $userQuery]]]
            ],
            "systemInstruction" => ["parts" => [["text" => $systemPrompt]]],
            "generationConfig" => [
                "temperature" => 0.3,
                "maxOutputTokens" => 1500
            ]
        ];

        $ch = curl_init($endpoint);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        
        $start = microtime(true);
        $response = curl_exec($ch);
        $latency = (int)((microtime(true) - $start) * 1000);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode !== 200) {
            return ['error' => 'Error API: ' . $httpCode, 'details' => $response];
        }

        $json = json_decode($response, true);
        $text = $json['candidates'][0]['content']['parts'][0]['text'] ?? 'Sin respuesta.';
        $usage = $json['usageMetadata'] ?? ['promptTokenCount' => 0, 'candidatesTokenCount' => 0];

        return [
            'text' => $text,
            'input_tokens' => $usage['promptTokenCount'],
            'output_tokens' => $usage['candidatesTokenCount'],
            'latency' => $latency,
            'status' => $httpCode
        ];
    }
}