<?php
// Ubicación: public_html/asesoria.virtualex.ec/app/Models/User.php

namespace App\Models;

use App\Config\Database;
use PDO;

class User {

    /**
     * Obtener usuario por ID
     */
    public static function find(int $id): ?array {
        $db = Database::getApp();
        $stmt = $db->prepare("SELECT id, nombre, email, status FROM users WHERE id = ?");
        $stmt->execute([$id]);
        $user = $stmt->fetch();
        return $user ?: null;
    }

    /**
     * Obtener saldo actual de créditos
     */
    public static function getBalance(int $userId): float {
        $db = Database::getApp();
        $stmt = $db->prepare("SELECT balance FROM credit_balance WHERE user_id = ?");
        $stmt->execute([$userId]);
        
        // Si no existe registro de balance, devolver 0 (o crear uno por defecto)
        return (float) $stmt->fetchColumn(); 
    }

    /**
     * Verificar credenciales para Login (Hash seguro)
     */
    public static function authenticate(string $email, string $password): ?array {
        $db = Database::getApp();
        $stmt = $db->prepare("SELECT id, nombre, password_hash, status FROM users WHERE email = ?");
        $stmt->execute([$email]);
        $user = $stmt->fetch();

        if ($user && password_verify($password, $user['password_hash'])) {
            if ($user['status'] != 1) return null; // Usuario inactivo
            unset($user['password_hash']); // No devolver el hash
            return $user;
        }

        return null;
    }

    /**
     * Registrar nuevo usuario (Opcional)
     */
    public static function create(string $nombre, string $email, string $password): bool {
        $db = Database::getApp();
        $hash = password_hash($password, PASSWORD_BCRYPT);
        
        try {
            $db->beginTransaction();
            
            // 1. Crear Usuario
            $stmt = $db->prepare("INSERT INTO users (nombre, email, password_hash) VALUES (?, ?, ?)");
            $stmt->execute([$nombre, $email, $hash]);
            $userId = $db->lastInsertId();

            // 2. Inicializar Balance en 0 (o créditos de regalo)
            $stmtBal = $db->prepare("INSERT INTO credit_balance (user_id, balance) VALUES (?, 3)"); // 3 créditos gratis
            $stmtBal->execute([$userId]);
            
            // 3. Registrar Transacción de regalo
            $stmtTrans = $db->prepare("INSERT INTO credit_transactions (user_id, type, amount, description) VALUES (?, 'DEPOSIT', 3, 'Bono Bienvenida')");
            $stmtTrans->execute([$userId]);

            $db->commit();
            return true;
        } catch (\Exception $e) {
            $db->rollBack();
            return false;
        }
    }
}